import logging
from typing import Optional, List

import pandas as pd

from .base import BaseFeatureGenerator


logging.basicConfig(level=20, format="%(asctime)s - %(levelname)s - %(message)s")
logger = logging.getLogger(__name__)


class ClipRatingFeatureGenerator(BaseFeatureGenerator):
    def get_nlp_data(self, clip_timestamps: list) -> Optional[pd.DataFrame]:
        sentence_info = (
            self.nlp.get_sentence(clip_timestamps[0], clip_timestamps[1])
            if clip_timestamps is not None
            else ("", 0)
        )
        sentence = sentence_info[0]

        # Length
        sentence_length = len(sentence)

        # Sentiment analysis
        sentence_sentiment = self.nlp.sentiment_scores(sentence)

        # Keywords correlation
        keywords_correlation = self.nlp.calculate_thematic_coherence(sentence)

        # Lexical diversity
        lexical_diversity = self.nlp.lexical_diversity(sentence)

        # Vertical syntactic dictionaries
        vertical_syntactic_dictionaries_mentions = self.nlp.get_vertical_syntactic_dictionaries_mentions(sentence)

        # Mention of brand
        is_brand_mentioned = self.nlp.is_brand_mentioned(sentence)

        # Numeric features
        has_numbers = self.nlp.has_numbers(sentence)
        percentage_mentioned = self.nlp.percentage_mentioned(sentence)
        price_mentioned = self.nlp.price_mentioned(sentence)

        data = {
            "sentence_length": sentence_length,
            "sentiment": sentence_sentiment["compound"],
            "current_keywords_correlation": keywords_correlation,
            "lexical_diversity": lexical_diversity,
            "is_brand_mentioned": is_brand_mentioned,
            "has_numbers": has_numbers,
            "percentage_mentioned": percentage_mentioned,
            "price_mentioned": price_mentioned,
            **vertical_syntactic_dictionaries_mentions,
        }

        if data:
            return pd.DataFrame(data, index=[0])

    def get_od_data(self, clip_timestamp: list, combined_categories: Optional[list]) -> Optional[pd.DataFrame]:
        frames = self.od.get_frames_ids(clip_timestamp[0], clip_timestamp[1])

        number_of_objects = self.od.get_number_of_objects(frames, combined_categories)

        state_changes_deviation = self.od.get_state_changes_deviation(
            frames, combined_categories
        )
        is_product_applied = self.od.is_product_applied(frames, combined_categories)
        percentage_product_is_visible = self.od.percentage_product_is_visible(frames, combined_categories)
        focus_on_face = self.od.is_focus_on_face(frames)
        focus_on_product = self.od.is_focus_on_product(frames)

        data = {
            "state_changes_deviation": state_changes_deviation,
            "is_product_applied": is_product_applied,
            "percentage_product_is_visible": percentage_product_is_visible,
            "focus_on_face": focus_on_face,
            "focus_on_product": focus_on_product,
        }
        data.update(number_of_objects)

        if data:
            return pd.DataFrame(data, index=[0])

    def get_features_for_single_clip(
        self, clip_timestamps: list, combined_categories_chosen: Optional[list] = None,
    ) -> Optional[pd.DataFrame]:
        combined_categories_chosen = combined_categories_chosen or []

        try:
            logger.info(f"Getting features for clip: {clip_timestamps}")
            nlp = self.get_nlp_data(clip_timestamps)
            od = self.get_od_data(clip_timestamps, combined_categories_chosen)
            if nlp is not None and od is not None:
                return pd.concat([nlp, od], axis=1, join="inner")

        except Exception as error:
            logger.error(f"Unexpected error in main function get_features: {error}")
            raise error

    def get_features_for_video_file(self, segments: list) -> List[dict]:
        try:
            results = []

            for segment in segments:
                start = float(segment["START"]) / 1000
                end = float(segment["END"]) / 1000
                current_segment = [start, end]
                features = self.get_features_for_single_clip(current_segment)

                if not features.empty:
                    features = features.to_json(orient="records")
                    results.append(
                        {
                            "segment": current_segment,
                            "features": features,
                            "video_clip_id": segment["video_clip_id"],
                        }
                    )

            return results
        except Exception as error:
            logger.error(f"Error retrieving features: {error}")
            raise error
